#!/usr/bin/perl -w
# This CGI script presents the results of the build_farm build
#
# Copyright (C) Andrew Tridgell <tridge@samba.org>     2001
# Copyright (C) Andrew Bartlett <abartlet@samba.org>   2001
# Copyright (C) Vance Lankhaar  <vance@pcsscreston.ca> 2002
# Copyright (C) Martin Pool <mbp@samba.org>            2001
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#   
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#   
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

# TODO: Allow filtering of the "Recent builds" list to show
# e.g. only broken builds or only builds that you care about.


my $BASEDIR = "/var/local/bluefish_build";

use strict qw{vars};
use lib "/var/www/html/bluefish_build";
use util;
use history;
use POSIX;
use Data::Dumper;
use CGI::Base;
use CGI::Form;
use File::stat;

my $req = new CGI::Form;

my $HEADCOLOR = "#a0a0e0";
my $OLDAGE = 60*60*4;
my $DEADAGE = 60*60*24*4;

##############################################
# this defines what it is possible to build 
# and what boxes. Should be in a config file
my $compilers = ['cc', 'gcc', 'gcc3', 'insure'];

my (%hosts) = ('trip' => "Mandrake 9.0, P4 1700" 
	       );


my @hosts = sort { $hosts{$a} cmp $hosts{$b} } keys %hosts;

my (%trees) = ('bluefish' => "",
	       'bluefish-gtk2' => "bluefish-gtk2" );

# this is automatically filled in
my (@deadhosts) = ();
my $broken = 0;


###############################################
# work out a URL so I can refer to myself in links
#my $mysite = $req->self_url;
#if ($mysite =~ /(.*)[?].*/) {
	#    $mysite = $1;
#}
#if ($mysite =~ /http:\/\/.*\/(.*)/) {
	#    $mysite = $1;
#}

my $mysite = "http://bluefish.mrball.net/cgi-bin/bluefish_build/build.pl";

################################################
# start CGI headers
sub cgi_headers() {
    print "Content-type: text/html\r\n";

    util::cgi_gzip();

    print '<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN" "http://www.w3.org/TR/html4/loose.dtd"> 
<html>
<head><title>Bluefish Build Farm</title></head>
<body bgcolor="white" text="#000000" link="#0000EE" vlink="#551A8B" alink="#FF0000">
<table border=0>
<tr>
<td><img alt="Bluefish Banner" border=0 align="left" src="http://bluefish.openoffice.nl/go1_1.png"></td>
<td>
<ul>
<li><a href="/bluefish_build/about.html">About the build farm</a>
<li><a href="/bluefish_build/instructions.html">Adding a new machine</a>
</ul>
</td>
</tr>
</table>
';

}

################################################
# end CGI
sub cgi_footers() {
    print "</body>";
    print "</html>\n";
}

################################################
# print an error on fatal errors
sub fatal($) {
    my $msg=shift;
    print "ERROR: $msg<br>\n";
    cgi_footers();
    exit(0);
}

##############################################
# get the age of build from ctime
sub build_age($$$)
{
    my $host=shift;
    my $tree=shift;
    my $compiler=shift;
    my $file="build.$tree.$host.$compiler";
    my $age = -1;
    my $st;

    if ($st = stat("$file.log")) {
	$age = time() - $st->ctime;
    }

    return $age;
}

#############################################
# get the overall age of a host 
sub host_age($)
{
	my $host = shift;
	my $ret = -1;
	for my $compiler (@{$compilers}) {
		for my $tree (keys %trees) {
			my $age = build_age($host, $tree, $compiler);
			if ($age != -1 && ($age < $ret || $ret == -1)) {
				$ret = $age;
			}
		}
	}
	return $ret;
}

#############################################
# show an age as a string
sub red_age($)
{
	my $age = shift;
	
	if ($age > $OLDAGE) { 
		return sprintf("<font color=\"#b00000\">%s</font>",  util::dhm_time($age));
	}
	return util::dhm_time($age);
}


##############################################
# get status of build
sub build_status($$$)
{
    my $host=shift;
    my $tree=shift;
    my $compiler=shift;
    my $file="build.$tree.$host.$compiler";
    my $cstatus = "?";
    my $bstatus = "?";
    my $istatus = "?";
    my $tstatus = "?";
    my $sstatus = "/?";

    my $log;
    my $ret;

    my $st1 = stat("$file.log");
    my $st2 = stat("$file.status");
    
    if ($st1 && $st2 && $st1->ctime <= $st2->ctime) {
	return util::FileLoad("$file.status");
    }

    $log = util::FileLoad("$file.log");

    if ($log =~ /TEST STATUS:(.*)/) {
	if ($1 == 0) {
	    $tstatus = "<font color=green>ok</font>";
	} else {
	    $tstatus = "<font color=red>$1</font>";
	}
    }
    
    if ($log =~ /INSTALL STATUS:(.*)/) {
	if ($1 == 0) {
	    $istatus = "<font color=green>ok</font>";
	} else {
	    $istatus = "<font color=red>$1</font>";
	}
    }
    
    if ($log =~ /BUILD STATUS:(.*)/) {
	if ($1 == 0) {
	    $bstatus = "<font color=green>ok</font>";
	} else {
	    $bstatus = "<font color=red>$1</font>";
	}
    }

    if ($log =~ /CONFIGURE STATUS:(.*)/) {
	if ($1 == 0) {
	    $cstatus = "<font color=green>ok</font>";
	} else {
	    $cstatus = "<font color=red>$1</font>";
	}
    }
    
    if ($log =~ /INTERNAL ERROR:(.*)/ || $log =~ /PANIC:(.*)/) {
	$sstatus = "/<font color=red><b>PANIC</b></font>";
    } else {
	$sstatus = "";
    }
    
    $ret = "<a href=\"$mysite?function=View+Build&host=$host&tree=$tree&compiler=$compiler\">$cstatus/$bstatus/$istatus/$tstatus$sstatus</a>";


    util::FileSave("$file.status", $ret);

    return $ret;
}


##############################################
# get status of build
sub err_count($$$)
{
    my $host=shift;
    my $tree=shift;
    my $compiler=shift;
    my $file="build.$tree.$host.$compiler";
    my $err;

    my $st1 = stat("$file.err");
    my $st2 = stat("$file.errcount");

    if ($st1 && $st2 && $st1->ctime <= $st2->ctime) {
	    return util::FileLoad("$file.errcount");
    }

    $err = util::FileLoad("$file.err");

    if (! $err) { return 0; }

    my $ret = util::count_lines($err);

    util::FileSave("$file.errcount", "$ret");

    return $ret;
}



##############################################
# view build summary
sub view_summary() {
    my $i = 0;
    my $list = `ls`;

    my $cols = 2;

    my $broken = 0;

    # set up counters
    my %broken_count;
    my %panic_count;
    my %host_count;

    # zero broken and panic counters
    for my $tree (keys %trees) {
	$broken_count{$tree} = 0;
	$panic_count{$tree} = 0;
	$host_count{$tree} = 0;
    }

    #set up a variable to store the broken builds table's code, so we can output when we want
    my $broken_table;

    my $host_os;
    my $last_host = "";

    for my $host (@hosts) {
	for my $compiler (@{$compilers}) {
	    for my $tree (keys %trees) {
		my $status = build_status($host, $tree, $compiler);
		my $age = build_age($host, $tree, $compiler);
		if ($age != -1 && $age < $DEADAGE) {
		    $host_count{$tree}++;
		}
		if ($age < $DEADAGE && $status =~ /color=red/) {
		    if (!$broken) {
			$broken_table .= sprintf "<b>Currently broken builds:</b><p>\n";
			$broken_table .= sprintf "<table border=2><tr
      bgcolor=\"$HEADCOLOR\"><th colspan=3>Target</th><th>Build&nbsp;Age</th><th>Status<br>config/build/install/test</th><th>warnings</th></tr>\n";
			$broken = 1;
		    }
		    $broken_count{$tree}++;
		    if ($status =~ /PANIC/) {
			$panic_count{$tree}++;
		    }
		    my $warnings = err_count($host, $tree, $compiler);
		    
		    $broken_table .= sprintf "<tr>";
		    
		    $host_os = $hosts{$host};
		    if ($host eq $last_host) {
			$broken_table .= sprintf "<td colspan=2></td>";
		    } else {
			$broken_table .= sprintf "<td>$host_os</td><td>$host</td>";
		    }
		    $broken_table .= sprintf "<td><b>$tree</b>/$compiler</td><td align=right>" . red_age($age) . "</td><td align=center>$status</td><td align=center>$warnings</td></tr>\n";
		    
		    $last_host = $host;
		    
		}
	    }
	}
    }
    
    if ($broken) {
	$broken_table .= sprintf("</table><p>\n");
    }

    print "<b>Build counts:</b><p>";
    print "<table border=2 width=250><tr bgcolor=\"$HEADCOLOR\"><th>Tree</th><th>Total</th><th>Broken</th><th>Panic</th></tr>\n";
    for my $tree (keys %trees) {
	print "<tr><td>$tree</td><td align=center>$host_count{$tree}</td><td align=center>$broken_count{$tree}</td><td align=center><font color=red><b>$panic_count{$tree}</b></font></td></tr>\n";
    }
    print "</table><p>\n";


    print $broken_table;

    print "<b>Build summary:</b>\n\n";
    
    print '<table border=0><tr>';
    for my $host (@hosts) {
	# make sure we have some data from it
	if (! ($list =~ /$host/)) { print "\n<!-- skipping $host --!>\n"; next; }
	
	if ($i == $cols) {
	    $i = 0;
	    print "</tr><tr>";
	}

	my $row = 0;
	
	for my $compiler (@{$compilers}) {
	    for my $tree (keys %trees) {
		my $age = build_age($host, $tree, $compiler);
		my $warnings = err_count($host, $tree, $compiler);
		if ($age != -1 && $age < $DEADAGE) {
		    my $status = build_status($host, $tree, $compiler);
		    if ($row == 0) {
			print "<td valign=top><br><b>$host - $hosts{$host}</b><br><table border=2>
<tr bgcolor=\"$HEADCOLOR\"><th>Target</th><th>Build&nbsp;Age</th><th>Status<br>config/build<br>install/test</th><th>warnings</th></tr>
";
		    }
		    print "<tr align=center><td align=left><b>$tree</b>/$compiler</td><td align=right>" . red_age($age) . "</td><td>$status</td><td>$warnings</td></tr>\n";
		    $row++;
		}
	    }
	}
	if ($row != 0) {
	    print "</table></td>\n";
	    $i++;
	} else {
	    push(@deadhosts, $host);
	}
    }
    print '</tr></table>';

    draw_dead_hosts(@deadhosts);
}

##############################################
# Draw the "recent builds" view

sub view_recent_builds() {
    my $i = 0;
    my $list = `ls`;

    my $cols = 2;

    my $host_os;
    my $last_host = "";
    my @all_builds = ();
    my $tree=$req->param("tree");

    # Convert from the DataDumper tree form to an array that 
    # can be sorted by time.

    for my $host (@hosts) {
      for my $compiler (@{$compilers}) {
	  my $status = build_status($host, $tree, $compiler);
	  my $age = build_age($host, $tree, $compiler);
	  push @all_builds, [$age, $hosts{$host}, $host, $compiler, $tree, $status]
	  	unless $age == -1 or $age >= $DEADAGE;
      }
  }

  @all_builds = sort {$$a[0] <=> $$b[0]} @all_builds;
  

    print "<h2>Recent builds of $tree</h2>";
    print '<table border=2>';
    print "<tr bgcolor=\"$HEADCOLOR\">";
    print "<th>Age</th>";
    print "<th colspan=4>Target</th>";
    print "<th>Status</th>";
    print "</tr>\n";

    for my $build (@all_builds) {
	my $age = $$build[0];
	printf "<tr>";
	print "<td>" .
	util::dhm_time($age)."<td>";		# goes straight to stdout
	print join "<td>", @$build[4, 1, 2, 3, 5];
	print "</tr>\n";
    }
    print "</table>\n";
}


##############################################
# Draw the "dead hosts" table
sub draw_dead_hosts() {
    my @deadhosts = @_;
    print "<br><b>Dead Hosts:</b><br>\n";
    print '<table border=2><tr>';
    print "<tr bgcolor=\"$HEADCOLOR\"><th>Host</th><th>OS</th><th>Min Age</th></tr>
";
    for my $host (@deadhosts) {
	my $age = host_age($host);
	printf("<tr><td>$host</td><td>$hosts{$host}</td><td align=right>%s</td>\n", util::dhm_time($age));
    }    
    print "</table>\n";
}


##############################################
# view one build in detail
sub view_build() {
    my $host=$req->param("host");
    my $tree=$req->param("tree");
    my $compiler=$req->param("compiler");
    my $file="build.$tree.$host.$compiler";
    my $log;
    my $err;
    my $uname="";
    my $cflags="";
    my $config="";
    my $age = build_age($host, $tree, $compiler);
    my $status = build_status($host, $tree, $compiler);

    util::InArray($host, [keys %hosts]) || fatal("unknown host");
    util::InArray($compiler, $compilers) || fatal("unknown compiler");
    util::InArray($tree, [keys %trees]) || fatal("unknown tree");

    $log = util::FileLoad("$file.log");
    $err = util::FileLoad("$file.err");
    
    if ($log) {
	$log = util::cgi_escape($log);

	if ($log =~ /(.*)/) { $uname=$1; }
	if ($log =~ /CFLAGS=(.*)/) { $cflags=$1; }
	if ($log =~ /configure options: (.*)/) { $config=$1; }
    }

    if ($err) {
	$err = util::cgi_escape($err);
    }

    print util::FileLoad("../web/$host.html");

    print "
<table>
<tr><td>Host:</td><td>$host - $hosts{$host}</td></tr>
<tr><td>Uname:</td><td>$uname</td></tr>
<tr><td>Tree:</td><td>$tree</td></tr>
<tr><td>Build age:</td><td>" . red_age($age) . "</td></tr>
<tr><td>Status:</td><td>$status</td></tr>
<tr><td>Compiler:</td><td>$compiler</td></tr>
<tr><td>CFLAGS:</td><td>$cflags</td></tr>
<tr><td>configure options:  </td><td>$config</td></tr>
</table>
";

    # These can be pretty wide -- perhaps we need to 
    # allow them to wrap in some way?
    if ($err eq "") {
	print "<b>No error log available</b><br>\n";
    } else {
	print "<h2>Error log:</h2>\n";
	print "<tt><pre>" . join('', $err) . "</pre></tt>\n";
    }

    if ($log eq "") {
	print "<b>No build log available</b><br>\n";
    } else {
	print "<h2>Build log:</h2>\n";
	print "<tt><pre>" . join('', $log) . "</pre></tt><p>\n";
    }

    print "</body>\n";
}


##############################################
# main page
sub main_menu() {
    print $req->startform("GET");
    print $req->popup_menu(-name=>'host',
			   -values=>\@hosts,
			   -labels=>\%hosts);
    print $req->popup_menu("tree", [keys %trees]);
    print $req->popup_menu("compiler", $compilers);
    
    print $req->submit('function', 'View Build');
    print "&nbsp;&nbsp;" . $req->submit('function', 'Recent Checkins');
    print "&nbsp;&nbsp;" . $req->submit('function', 'Summary');
    print "&nbsp;&nbsp;" . $req->submit('function', 'Recent Builds');

    print $req->endform();
}


###############################################
# main program
cgi_headers();

chdir("$BASEDIR") || fatal("can't change to data directory");

main_menu();

if (defined $req->param("function")) {
    my $fn_name = $req->param("function");
    if ($fn_name eq "View Build") {
	view_build();
    } elsif ($fn_name eq "Recent Builds") {
	view_recent_builds();
    } elsif ($fn_name eq "Recent Checkins") {
	history::cvs_history($req->param('tree'));
    } elsif ($fn_name eq "diff") {
	history::cvs_diff($req->param('author'), $req->param('date'), $req->param('tree'), "html");
    } elsif ($fn_name eq "text_diff") {
	history::cvs_diff($req->param('author'), $req->param('date'), $req->param('tree'), "text");
    } else {
	view_summary();
    }
} else {
    view_summary();
}

cgi_footers();
